import { getServerSession } from "next-auth";
import { authOptions } from "@/app/api/auth/[...nextauth]/route";
import { redirect } from "next/navigation";
import { getAnnonceById, getAnnonceByIdInclude } from "@/lib/services/annonceService";
import { getUserById } from "@/lib/services/userService";
import Link from "next/link";
import ImageCarousel from "@/lib/components/ImageCarrousel";
import ImageCarrousel from "@/lib/components/ImageCarrousel";

export default async function AnnonceDetail({ params }: { params: { id: string } }) {
    const session = await getServerSession(authOptions);
    if (!session) redirect("/");

    const annonce = await getAnnonceByIdInclude(Number(params.id));
    if (!annonce) return <div className="p-20 text-center">Annonce non trouvée.</div>;

    const agent = await getUserById(annonce.agentId);
    const formattedPrice = new Intl.NumberFormat("fr-FR", {
        style: "currency",
        currency: "EUR",
        minimumFractionDigits: 0,
    }).format(annonce.prix);
    const imageSrc = annonce.imagePrincipale
        ? `data:image/jpeg;base64,${annonce.imagePrincipale}`
        : "/default-image-annonce.jpg";

    let imagesSrc = [];

    if (annonce.autresImages && annonce.autresImages.length > 0) {
        const autresImagesSrc = annonce.autresImages.map((img) => `data:image/jpeg;base64,${img.imageData}`);
        imagesSrc = [imageSrc, ...autresImagesSrc];
    } else {
        imagesSrc = [imageSrc];
    }
    return (
        <div className="min-h-screen bg-white">
            <div className="container mx-auto p-4 sm:p-6 lg:p-8">
                <header className="mb-4 py-4">
                    <Link
                        href="/"
                        className="mb-8 inline-flex items-center text-sm font-medium text-gray-500 hover:text-black">
                        &larr; Retour aux annonces
                    </Link>
                </header>

                <main className="grid grid-cols-1 lg:grid-cols-3 lg:gap-12">
                    {/* Colonne principale (Image et Description) */}
                    <div className="lg:col-span-2">
                        {/* Carrousel */}
                        <div className="mb-8">
                            <ImageCarrousel images={imagesSrc} titre={annonce.titre} />
                        </div>

                        {/* Titre et Adresse */}
                        <div className="mb-6 border-b border-gray-200 pb-6">
                            <h1 className="text-4xl font-bold text-gray-900">{annonce.titre}</h1>
                            <p className="text-md mt-2 text-gray-500">
                                {annonce.address}, {annonce.ville}
                            </p>
                        </div>

                        {/* Description */}
                        <h2 className="mb-4 text-2xl font-semibold text-gray-800">Description</h2>
                        <p className="leading-relaxed text-gray-600">{annonce.description || "Aucune description."}</p>
                    </div>

                    {/* Prix et Agent */}
                    <aside className="mt-8 lg:mt-0">
                        <div className="sticky top-8 rounded-xl border border-gray-200 bg-white p-6 shadow-sm">
                            <p className="mb-6 text-3xl font-bold text-gray-900">{formattedPrice}</p>

                            <h3 className="border-t border-gray-200 pt-6 text-xl font-semibold text-gray-800">
                                Agent en charge
                            </h3>
                            {agent ? (
                                <div className="mt-4 space-y-3">
                                    <p className="text-gray-600">
                                        <span className="font-medium text-gray-900">Nom:</span> {agent.firstName}{" "}
                                        {agent.lastName}
                                    </p>
                                    <p className="text-gray-600">
                                        <span className="font-medium text-gray-900">Email:</span> {agent.email}
                                    </p>
                                    <button className="mt-4 w-full rounded-md bg-gray-900 px-4 py-2 text-sm font-medium text-white hover:bg-black">
                                        Contacter l'agent
                                    </button>
                                </div>
                            ) : (
                                <p className="mt-4 text-gray-600">Information indisponible.</p>
                            )}
                        </div>
                    </aside>
                </main>
            </div>
        </div>
    );
}
